// Komunikacija po protokolu gRPC
// odjemalec

package main

import (
	"api/grpc/protobufStorage"
	"context"
	"fmt"
	"io"
	"sync"
	"time"

	"google.golang.org/grpc"
	"google.golang.org/grpc/credentials/insecure"
	"google.golang.org/protobuf/types/known/emptypb"
)

func Client(url string) {
	// vzpostavimo povezavo s strežnikom
	fmt.Printf("gRPC client connecting to %v\n", url)
	conn, err := grpc.NewClient(url, grpc.WithTransportCredentials(insecure.NewCredentials()))
	if err != nil {
		panic(err)
	}
	defer conn.Close()

	// vzpostavimo vmesnik gRPC
	grpcClient := protobufStorage.NewCRUDClient(conn)

	// uporabimo WaitGroup za sinhronizacijo gorutin
	var wg sync.WaitGroup
	wg.Add(2)

	// Gorutina 1: Spremembe v shrambi
	go func() {
		defer wg.Done()

		contextCRUD, cancel := context.WithTimeout(context.Background(), 30*time.Second)
		defer cancel()

		// Pripravimo strukture, ki jih uporabljamo kot argumente
		lecturesCreate := protobufStorage.Todo{Task: "predavanja", Completed: false}
		lecturesUpdate := protobufStorage.Todo{Task: "predavanja", Completed: true}
		practicals := protobufStorage.Todo{Task: "vaje", Completed: false}
		homeworkCreate := protobufStorage.Todo{Task: "domaca naloga", Completed: true}

		// Create - predavanja
		fmt.Println("\n[ACTION] Creating: predavanja")
		if _, err := grpcClient.Create(contextCRUD, &lecturesCreate); err != nil {
			fmt.Printf("Error creating: %v\n", err)
		}
		time.Sleep(1 * time.Second)

		// Create - vaje
		fmt.Println("[ACTION] Creating: vaje")
		if _, err := grpcClient.Create(contextCRUD, &practicals); err != nil {
			fmt.Printf("Error creating: %v\n", err)
		}
		time.Sleep(1 * time.Second)

		// Update - predavanja
		fmt.Println("[ACTION] Updating: predavanja")
		if _, err := grpcClient.Update(contextCRUD, &lecturesUpdate); err != nil {
			fmt.Printf("Error updating: %v\n", err)
		}
		time.Sleep(1 * time.Second)

		// Delete - vaje
		fmt.Println("[ACTION] Deleting: vaje")
		if _, err := grpcClient.Delete(contextCRUD, &practicals); err != nil {
			fmt.Printf("Error deleting: %v\n", err)
		}
		time.Sleep(1 * time.Second)

		// Create - domaca naloga
		fmt.Println("[ACTION] Creating: domaca naloga")
		if _, err := grpcClient.Create(contextCRUD, &homeworkCreate); err != nil {
			fmt.Printf("Error creating: %v\n", err)
		}
		time.Sleep(1 * time.Second)

		fmt.Println("[ACTION] All operations completed")
	}()

	// Gorutina 2: Spremljanje dogodkov
	go func() {
		defer wg.Done()

		contextSubscribe := context.Background()

		fmt.Println("\n[SUBSCRIBE] Starting to listen for events...")
		stream, err := grpcClient.Subscribe(contextSubscribe, &emptypb.Empty{})
		if err != nil {
			fmt.Printf("Error subscribing: %v\n", err)
			return
		}

		for {
			event, err := stream.Recv()
			if err == io.EOF {
				fmt.Println("[SUBSCRIBE] Stream ended")
				break
			}
			if err != nil {
				fmt.Printf("Error receiving event: %v\n", err)
				break
			}
			fmt.Printf("[EVENT] Action: %s | Task: %s | Completed: %v\n",
				event.Action, event.T.Task, event.T.Completed)
		}
	}()

	wg.Wait()
	fmt.Println("\nClient completed")
}
